package de.davidartmann.rosa.fragment;

import android.Manifest;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.design.widget.FloatingActionButton;
import android.support.design.widget.Snackbar;
import android.support.v4.app.ActivityCompat;
import android.support.v4.app.Fragment;
import android.support.v7.app.AlertDialog;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.view.animation.AccelerateDecelerateInterpolator;
import android.widget.ArrayAdapter;
import android.widget.ImageView;
import android.widget.TextView;

import com.squareup.picasso.Picasso;

import java.io.File;
import java.util.List;

import de.davidartmann.rosa.R;
import de.davidartmann.rosa.activity.MainListActivity;
import de.davidartmann.rosa.db.async.DecrementPositionOfAllPersonsFromDb;
import de.davidartmann.rosa.db.async.FindAllActivePersonsFromDb;
import de.davidartmann.rosa.db.async.SoftDeletePersonInDb;
import de.davidartmann.rosa.db.async.UpdatePersonInDb;
import de.davidartmann.rosa.model.Person;
import de.davidartmann.rosa.util.RoundedTransformation;

/**
 * Fragment for the detail view of a person.
 * Created by david on 03.03.16.
 */
public class DetailFragment extends Fragment implements View.OnClickListener,
        UpdatePersonInDb.IUpdatePersonInDb, MenuItem.OnMenuItemClickListener,
        FindAllActivePersonsFromDb.IFindAllActive,
        DecrementPositionOfAllPersonsFromDb.IDecrementPosition,
        SoftDeletePersonInDb.ISoftDeletePerson {

    public static final String PERSON = "person";
    public static final String EDIT = "edit";
    public static final int REQUEST_PERMISSION_CALL_PHONE = 0;

    private TextView textViewName;
    private TextView textViewPhone;
    private TextView textViewEmail;
    private TextView textViewAddress;
    private TextView textViewPrice;
    private TextView textViewMisc;
    private ImageView imageViewPicture;
    private Person mPerson;
    private int mDecrementStartPos;
    private FloatingActionButton mFab;

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container,
                             Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.fragment_detail, container, false);
        textViewName = (TextView) view.findViewById(R.id.fragment_detail_name);
        textViewPhone = (TextView) view.findViewById(R.id.fragment_detail_phone);
        textViewEmail = (TextView) view.findViewById(R.id.fragment_detail_email);
        textViewAddress = (TextView) view.findViewById(R.id.fragment_detail_address);
        textViewPrice = (TextView) view.findViewById(R.id.fragment_detail_price);
        textViewMisc = (TextView) view.findViewById(R.id.fragment_detail_misc);
        imageViewPicture = (ImageView) view.findViewById(R.id.fragment_detail_picture);
        mFab = (FloatingActionButton) view.findViewById(R.id.fragment_detail_fab);
        mFab.setOnClickListener(this);
        mPerson = (Person) getArguments().getSerializable(PERSON);
        setHasOptionsMenu(true);
        assignData(mPerson);
        mDecrementStartPos = 0;
        if (mPerson.getPhone() != null) {
            if (!mPerson.getPhone().isEmpty()) {
                textViewPhone.setOnClickListener(this);
            }
        }
        if (mPerson.getAddress() != null) {
            if (!mPerson.getAddress().isEmpty()) {
                textViewAddress.setOnClickListener(this);
            }
        }
        if (mPerson.getEmail() != null) {
            if (!mPerson.getEmail().isEmpty()) {
                textViewEmail.setOnClickListener(this);
            }
        }
        return view;
    }

    @Override
    public void onCreateOptionsMenu(Menu menu, MenuInflater inflater) {
        super.onCreateOptionsMenu(menu, inflater);
        inflater.inflate(R.menu.menu_detail, menu);
        MenuItem deleteMenuItem = menu.findItem(R.id.menu_detail_delete);
        MenuItem editMenuItem = menu.findItem(R.id.menu_detail_edit);
        deleteMenuItem.setOnMenuItemClickListener(this);
        editMenuItem.setOnMenuItemClickListener(this);
    }

    public static DetailFragment newInstance(Person person) {
        DetailFragment detailFragment = new DetailFragment();
        Bundle bundle = new Bundle();
        bundle.putSerializable(PERSON, person);
        detailFragment.setArguments(bundle);
        return detailFragment;
    }

    private void assignData(Person person) {
        Context context = getContext();
        textViewName.setText(person.getName());
        textViewPhone.setText(person.getPhone());
        textViewEmail.setText(person.getEmail());
        textViewAddress.setText(person.getAddress());
        textViewPrice.setText(person.getPrice());
        textViewMisc.setText(person.getMisc());
        int dimen;
        if (context.getResources().getConfiguration().orientation
                == Configuration.ORIENTATION_PORTRAIT) {
            dimen = R.dimen.imageview_detail_dimen_portrait;
        } else {
            dimen = R.dimen.imageview_detail_dimen_landscape;
        }
        File file = null;
        if (person.getPictureUrl() != null) {
            file = new File(person.getPictureUrl());
        }
        Picasso.with(context)
                .load(file)
                .placeholder(R.drawable.user)
                .error(R.drawable.user)
                .transform(new RoundedTransformation(0, dimen))
                .resizeDimen(dimen, dimen)
                .centerCrop()
                .into(imageViewPicture);
    }

    @Override
    public void onClick(View v) {
        Context context = getContext();
        Intent intent;
        switch (v.getId()) {
            case R.id.fragment_detail_fab:
                showCategoryPicker();
                break;
            case R.id.fragment_detail_email:
                intent = new Intent(Intent.ACTION_SEND);
                intent.setType("text/plain");
                intent.putExtra(Intent.EXTRA_EMAIL, mPerson.getEmail());
                intent.putExtra(Intent.EXTRA_SUBJECT, "");
                intent.putExtra(Intent.EXTRA_TEXT, "");
                context.startActivity(Intent.createChooser(intent, "Email versenden mit"));
                break;
            case R.id.fragment_detail_phone:
                String phoneNr = mPerson.getPhone();
                if (phoneNr.startsWith("0", 0)) {
                    phoneNr = phoneNr.replaceFirst("0", "+49");
                }
                Uri uri = Uri.parse("smsto:" + phoneNr);
                Intent i = new Intent(Intent.ACTION_SENDTO, uri);
                boolean isWhatsappInstalled;
                try {
                    getContext().getPackageManager().getPackageInfo("com.whatsapp", PackageManager.GET_META_DATA);
                    isWhatsappInstalled = true;
                } catch (PackageManager.NameNotFoundException e) {
                    //e.printStackTrace();
                    Snackbar.make(v, "Whatsapp ist nicht installiert", Snackbar.LENGTH_LONG)
                            .show();
                    isWhatsappInstalled = false;
                }
                if (isWhatsappInstalled) {
                    i.setPackage("com.whatsapp");
                    startActivity(Intent.createChooser(i, "Teilen mit"));
                }
                /*
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                    if (ActivityCompat
                            .checkSelfPermission(context, Manifest.permission.CALL_PHONE)
                            != PackageManager.PERMISSION_GRANTED) {
                        ActivityCompat.requestPermissions(getActivity(),
                                new String[] { Manifest.permission.CALL_PHONE },
                                REQUEST_PERMISSION_CALL_PHONE);
                    } else {
                        makeCallIntent(context);
                    }
                } else {
                    makeCallIntent(context);
                }
                */
                break;
            case R.id.fragment_detail_address:
                intent = new Intent(Intent.ACTION_VIEW,
                        Uri.parse("geo:0,0?q="+mPerson.getAddress()));
                context.startActivity(Intent.createChooser(intent, "Navigieren mit"));
                break;
        }
    }

    private void makeCallIntent(Context context) {
        Intent intent = new Intent(Intent.ACTION_CALL,
                Uri.parse("tel:" + mPerson.getPhone().trim()));
        context.startActivity(Intent.createChooser(intent, "Anrufen mit"));
    }

    private void showCategoryPicker() {
        final Context context = getContext();
        AlertDialog.Builder builder = new AlertDialog.Builder(context);
        builder.setTitle(getString(R.string.Wähle_Kategorie));
        final ArrayAdapter<String> arrayAdapter = new ArrayAdapter<>(context,
                android.R.layout.simple_list_item_single_choice);
        arrayAdapter.addAll(getResources().getStringArray(R.array.categories));
        builder.setSingleChoiceItems(arrayAdapter, mPerson.getCategory(),
                new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                // when category is changed, we want to store this new info
                new UpdatePersonInDb(context, DetailFragment.this)
                        .execute(setNewCategoryOfPerson(which));
                dialog.dismiss();
            }
        });
        builder.setNegativeButton("Abbrechen", new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                dialog.dismiss();
            }
        });
        builder.show();
    }

    private Person setNewCategoryOfPerson(int which) {
        mPerson.setCategory(which);
        return mPerson;
    }

    @Override
    public void onUpdate(Person person) {
        mPerson = person;
        assignData(person);
        mFab.animate().setDuration(200)
                .setInterpolator(new AccelerateDecelerateInterpolator())
                .rotation(45).withEndAction(new Runnable() {
            @Override
            public void run() {
                mFab.animate().setDuration(400)
                        .setInterpolator(new AccelerateDecelerateInterpolator())
                        .rotation(-45).withEndAction(new Runnable() {
                    @Override
                    public void run() {
                        mFab.animate().setDuration(200)
                                .setInterpolator(new AccelerateDecelerateInterpolator())
                                .rotation(0);
                    }
                });
            }
        });
    }

    @Override
    public boolean onMenuItemClick(MenuItem item) {
        switch (item.getItemId()) {
            case R.id.menu_detail_delete:
                showDeletionDialog();
                return true;
            case R.id.menu_detail_edit:
                showDetailEditFragment();
                return true;
        }
        return false;
    }

    private void showDetailEditFragment() {
        Context context = getContext();
        Intent intent = new Intent(context, MainListActivity.class);
        intent.putExtra(EDIT, true);
        intent.putExtra(PERSON, mPerson);
        //TODO: check this out, what is better destroy this instance or not -> also in the showDeletionDialog!
        //getActivity().finish();
        context.startActivity(intent);
    }

    private void showDeletionDialog() {
        final Context context = getContext();
        AlertDialog.Builder builder = new AlertDialog.Builder(context);
        builder.setMessage(context.getString(R.string.dialog_delete_message));
        builder.setPositiveButton(context.getString(R.string.Ja), new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                mDecrementStartPos = mPerson.getPosition();
                new SoftDeletePersonInDb(context, DetailFragment.this).execute(mPerson);
                dialog.dismiss();
            }
        });
        builder.setNegativeButton(context.getString(R.string.Nein), new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                dialog.dismiss();
            }
        });
        builder.show();
    }

    @Override
    public void onDecremented(List<Person> persons) {
        Context context = getContext();
        Intent intent = new Intent(context, MainListActivity.class);
        //getActivity().finish();
        context.startActivity(intent);
    }

    @Override
    public void onSoftDeleted(Person person) {
        new FindAllActivePersonsFromDb(getContext(), this).execute();
    }

    @Override
    public void onFindAllActive(List<Person> persons) {
        new DecrementPositionOfAllPersonsFromDb(getContext(), this, persons, mDecrementStartPos).execute();
    }
}
